<?php
/**
 * Scheduler for Image Compression Plugin
 * Handles automated compression scheduling
 */

if (!defined('ABSPATH')) {
    exit;
}

class ICP_Scheduler {
    
    private $compressor;
    private $hook_name = 'icp_scheduled_compression';
    
    public function __construct() {
        $this->compressor = new ICP_Compressor();
        $this->init_hooks();
    }
    
    private function init_hooks() {
        // Register the cron hook
        add_action($this->hook_name, array($this, 'run_scheduled_compression'));
        
        // Add filter to WordPress cron schedules to add our custom intervals
        add_filter('cron_schedules', array($this, 'add_custom_cron_schedules'));
        
        // AJAX handlers for scheduler management
        add_action('wp_ajax_icp_update_schedule', array($this, 'ajax_update_schedule'));
        add_action('wp_ajax_icp_run_scheduled_compression_now', array($this, 'ajax_run_scheduled_compression_now'));
        add_action('wp_ajax_icp_get_schedule_status', array($this, 'ajax_get_schedule_status'));
    }
    
    /**
     * Add custom cron schedules
     */
    public function add_custom_cron_schedules($schedules) {
        // Every minute (for testing - not recommended for production)
        $schedules['every_minute'] = array(
            'interval' => 60,
            'display' => __('Every Minute', 'image-compression-plugin')
        );
        
        // Every 5 minutes
        $schedules['every_5_minutes'] = array(
            'interval' => 300,
            'display' => __('Every 5 Minutes', 'image-compression-plugin')
        );
        
        // Every 15 minutes
        $schedules['every_15_minutes'] = array(
            'interval' => 900,
            'display' => __('Every 15 Minutes', 'image-compression-plugin')
        );
        
        // Every 30 minutes
        $schedules['every_30_minutes'] = array(
            'interval' => 1800,
            'display' => __('Every 30 Minutes', 'image-compression-plugin')
        );
        
        return $schedules;
    }
    
    /**
     * Schedule the compression job
     */
    public function schedule_compression($interval = 'hourly') {
        // Clear any existing scheduled event first
        $this->unschedule_compression();
        
        // Validate interval
        $valid_intervals = array(
            'every_minute', 'every_5_minutes', 'every_15_minutes', 'every_30_minutes',
            'hourly', 'twicedaily', 'daily', 'weekly', 'monthly'
        );
        
        if (!in_array($interval, $valid_intervals)) {
            $interval = 'hourly';
        }
        
        // Schedule the event
        if (!wp_next_scheduled($this->hook_name)) {
            wp_schedule_event(time(), $interval, $this->hook_name);
            
            // Log the scheduling
            $this->log_scheduler_activity('scheduled', "Compression scheduled for interval: $interval");
            
            return true;
        }
        
        return false;
    }
    
    /**
     * Unschedule the compression job
     */
    public function unschedule_compression() {
        $timestamp = wp_next_scheduled($this->hook_name);
        if ($timestamp) {
            wp_unschedule_event($timestamp, $this->hook_name);
            $this->log_scheduler_activity('unscheduled', 'Compression scheduling stopped');
            return true;
        }
        return false;
    }
    
    /**
     * Run the scheduled compression
     */
    public function run_scheduled_compression() {
        // Check if scheduling is enabled
        $settings = get_option('icp_scheduler_settings', array());
        if (!isset($settings['enabled']) || !$settings['enabled']) {
            return;
        }
        
        // Check if API key is configured
        $general_settings = get_option('icp_settings');
        $api_key = isset($general_settings['api_key']) ? trim($general_settings['api_key']) : '';
        
        if (empty($api_key)) {
            $this->log_scheduler_activity('error', 'Scheduled compression skipped: API key not configured');
            return;
        }
        
        // Get batch size from settings (default 5)
        $batch_size = isset($settings['batch_size']) ? intval($settings['batch_size']) : 5;
        
        // Handle unlimited batch processing (-1 means no limit)
        if ($batch_size === -1) {
            $batch_size = PHP_INT_MAX; // Process all images at once
        } else {
            $batch_size = max(1, $batch_size); // Minimum 1, no maximum limit
        }
        
        // Get excluded folders from settings
        $excluded_folders = isset($settings['excluded_folders']) ? $settings['excluded_folders'] : array();
        
        // Log folder exclusions if any
        if (!empty($excluded_folders) && ICP_Database::is_filebird_active()) {
            $excluded_count = count($excluded_folders);
            $this->log_scheduler_activity('info', "Folder exclusion active: {$excluded_count} folders excluded from compression");
        }
        
        // Get uncompressed images (excluding those in excluded folders)
        $uncompressed_images = ICP_Database::get_uncompressed_images($excluded_folders);
        
        if (empty($uncompressed_images)) {
            $this->log_scheduler_activity('info', 'No uncompressed images found');
            return;
        }
        
        // Process a batch of images
        $batch_images = array_slice($uncompressed_images, 0, $batch_size);
        $processed = 0;
        $successful = 0;
        $errors = array();
        
        foreach ($batch_images as $image) {
            $file_path = get_attached_file($image->ID);
            
            if (!$file_path || !file_exists($file_path)) {
                $errors[] = "File not found for attachment ID: {$image->ID}";
                continue;
            }
            
            // Set a flag to prevent conflicts with manual compression
            set_transient('icp_scheduling_' . $image->ID, true, 300); // 5 minutes
            
            $result = $this->compressor->compress_image_file($file_path, $image->ID);
            
            delete_transient('icp_scheduling_' . $image->ID);
            
            $processed++;
            
            if ($result && $result['success']) {
                $successful++;
            } elseif ($result) {
                $errors[] = "Attachment ID {$image->ID}: " . $result['error'];
            } else {
                $errors[] = "Unknown error compressing attachment ID: {$image->ID}";
            }
        }
        
        // Log the results
        $batch_info = ($settings['batch_size'] === -1) ? "unlimited batch" : "batch size {$settings['batch_size']}";
        $message = "Scheduled compression completed ({$batch_info}): {$successful}/{$processed} images compressed successfully";
        if (!empty($errors)) {
            $message .= ". Errors: " . implode(', ', array_slice($errors, 0, 3)); // Log first 3 errors
        }
        
        $this->log_scheduler_activity($successful > 0 ? 'success' : 'warning', $message);
        
        // Update last run time
        $settings['last_run'] = current_time('mysql');
        $settings['last_run_stats'] = array(
            'processed' => $processed,
            'successful' => $successful,
            'errors' => count($errors)
        );
        update_option('icp_scheduler_settings', $settings);
    }
    
    /**
     * Get scheduling status
     */
    public function get_schedule_status() {
        $settings = get_option('icp_scheduler_settings', array());
        $next_run = wp_next_scheduled($this->hook_name);
        
        return array(
            'enabled' => isset($settings['enabled']) ? $settings['enabled'] : false,
            'interval' => isset($settings['interval']) ? $settings['interval'] : 'hourly',
            'batch_size' => isset($settings['batch_size']) ? $settings['batch_size'] : 5,
            'excluded_folders' => isset($settings['excluded_folders']) ? $settings['excluded_folders'] : array(),
            'next_run' => $next_run ? wp_date('Y-m-d H:i:s', $next_run) : null,
            'last_run' => isset($settings['last_run']) ? $settings['last_run'] : null,
            'last_run_stats' => isset($settings['last_run_stats']) ? $settings['last_run_stats'] : null,
            'is_scheduled' => (bool) $next_run
        );
    }
    
    /**
     * Update schedule settings
     */
    public function update_schedule_settings($new_settings) {
        $current_settings = get_option('icp_scheduler_settings', array());
        $settings = wp_parse_args($new_settings, $current_settings);
        
        // Validate settings
        $settings['enabled'] = isset($settings['enabled']) ? (bool) $settings['enabled'] : false;
        
        // Handle batch size validation (-1 for unlimited, otherwise minimum 1)
        if (isset($settings['batch_size'])) {
            $batch_size = intval($settings['batch_size']);
            if ($batch_size === -1) {
                $settings['batch_size'] = -1; // Unlimited
            } else {
                $settings['batch_size'] = max(1, $batch_size); // Minimum 1, no maximum limit
            }
        } else {
            $settings['batch_size'] = 5; // Default
        }
        
        $valid_intervals = array(
            'every_minute', 'every_5_minutes', 'every_15_minutes', 'every_30_minutes',
            'hourly', 'twicedaily', 'daily', 'weekly'
        );
        
        if (!isset($settings['interval']) || !in_array($settings['interval'], $valid_intervals)) {
            $settings['interval'] = 'hourly';
        }
        
        // Validate excluded folders
        if (isset($settings['excluded_folders'])) {
            if (is_array($settings['excluded_folders'])) {
                // Filter out invalid folder IDs and convert to integers
                $settings['excluded_folders'] = array_filter(
                    array_map('intval', $settings['excluded_folders']),
                    function($id) { return $id > 0; }
                );
            } else {
                $settings['excluded_folders'] = array();
            }
        } else {
            $settings['excluded_folders'] = array();
        }
        
        // Update settings
        update_option('icp_scheduler_settings', $settings);
        
        // Update scheduling
        if ($settings['enabled']) {
            $this->schedule_compression($settings['interval']);
        } else {
            $this->unschedule_compression();
        }
        
        return $settings;
    }
    
    /**
     * Log scheduler activity
     */
    private function log_scheduler_activity($type, $message) {
        $logs = get_option('icp_scheduler_logs', array());
        
        $logs[] = array(
            'timestamp' => current_time('mysql'),
            'type' => $type,
            'message' => $message
        );
        
        // Keep only last 50 log entries
        if (count($logs) > 50) {
            $logs = array_slice($logs, -50);
        }
        
        update_option('icp_scheduler_logs', $logs);
    }
    
    /**
     * Get scheduler logs
     */
    public function get_scheduler_logs($limit = 20) {
        $logs = get_option('icp_scheduler_logs', array());
        return array_slice(array_reverse($logs), 0, $limit);
    }
    
    /**
     * Clear scheduler logs
     */
    public function clear_scheduler_logs() {
        delete_option('icp_scheduler_logs');
    }
    
    /**
     * AJAX handler for updating schedule
     */
    public function ajax_update_schedule() {
        check_ajax_referer('icp_scheduler_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $enabled = isset($_POST['enabled']) && $_POST['enabled'] === 'true';
        $interval = sanitize_text_field($_POST['interval']);
        $batch_size = intval($_POST['batch_size']);
        
        // Handle excluded folders
        $excluded_folders = array();
        if (isset($_POST['excluded_folders']) && is_array($_POST['excluded_folders'])) {
            $excluded_folders = array_filter(array_map('intval', $_POST['excluded_folders']));
        }
        
        $settings = $this->update_schedule_settings(array(
            'enabled' => $enabled,
            'interval' => $interval,
            'batch_size' => $batch_size,
            'excluded_folders' => $excluded_folders
        ));
        
        wp_send_json_success(array(
            'message' => $enabled ? 'Scheduled compression enabled' : 'Scheduled compression disabled',
            'settings' => $settings,
            'status' => $this->get_schedule_status()
        ));
    }
    
    /**
     * AJAX handler for running scheduled compression manually
     */
    public function ajax_run_scheduled_compression_now() {
        check_ajax_referer('icp_scheduler_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        // Force enable temporarily to run the compression
        $current_settings = get_option('icp_scheduler_settings', array());
        $was_enabled = isset($current_settings['enabled']) ? $current_settings['enabled'] : false;
        
        $current_settings['enabled'] = true;
        update_option('icp_scheduler_settings', $current_settings);
        
        // Run the compression
        $this->run_scheduled_compression();
        
        // Restore original enabled state
        $current_settings['enabled'] = $was_enabled;
        update_option('icp_scheduler_settings', $current_settings);
        
        wp_send_json_success(array(
            'message' => 'Manual compression run completed',
            'status' => $this->get_schedule_status()
        ));
    }
    
    /**
     * AJAX handler for getting schedule status
     */
    public function ajax_get_schedule_status() {
        check_ajax_referer('icp_scheduler_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        wp_send_json_success(array(
            'status' => $this->get_schedule_status(),
            'logs' => $this->get_scheduler_logs(10)
        ));
    }
} 